// naumann@stce.rwth-aachen.de

#include<cmath>
#include<stack>

enum mode { primal, adjoint };

template<typename T>
void f1(const T &x, T &y) {
  y=sin(x);
}

template<typename T>
void f1_t(const T &x, const T &x_t, T &y, T &y_t) {
  y_t=cos(x)*x_t;
  y=sin(x);
}

template<typename T>
void f1_t_a(mode m, const T &x, T &x_a, const T &x_t, T &x_t_a, T &y, T &y_a, T &y_t, T &y_t_a) {
  if (m==mode::primal) {
    y_t=cos(x)*x_t;
    y=sin(x);
  } else {
    x_a+=cos(x)*y_a; y_a=0;
    x_a+=-sin(x)*x_t*y_t_a; x_t_a+=cos(x)*y_t_a; y_t_a=0;
  }
}

template<typename T>
void f2(const T &x, T &y) {
  y=cos(x);
}

template<typename T>
void f2_t(const T &x, const T &x_t, T &y, T &y_t) {
  y_t=-sin(x)*x_t;
  y=cos(x);
}

template<typename T>
void f2_t_a(mode m, const T &x, T &x_a, const T &x_t, T &x_t_a, T &y, T &y_a, T &y_t, T &y_t_a) {
  if (m==mode::primal) {
    y_t=-sin(x)*x_t;
    y=cos(x);
  } else {
    x_a+=-sin(x)*y_a; y_a=0;
    x_a+=-cos(x)*x_t*y_t_a; x_t_a+=-sin(x)*y_t_a; y_t_a=0;
  }
}

template<typename T, typename PT>
void sigmoid(T &x, const PT &p, const PT &w) {
  T a; f1(x,a);
  T b; f2(x,b);
  x=1/(1+exp(-(x-p)/w));
  x=a*(1-x)+b*x;
}

template<typename T, typename PT>
void sigmoid_t(T &x, T &x_t, const PT &p, const PT &w) {
  T a, a_t;
  f1_t(x,x_t,a,a_t);
  T b, b_t;
  f2_t(x,x_t,b,b_t);
  T c_t=-exp(-(x-p)/w)/w*x_t;
  T c=1+exp(-(x-p)/w);
  x_t=-c_t/pow(c,2);
  x=1/c;
  x_t=(1-x)*a_t+(b-a)*x_t+x*b_t;
  x=a*(1-x)+b*x;
}

template<typename T, typename PT>
void sigmoid_t_a(mode m, T &x, T &x_a, T &x_t, T &x_t_a, const PT &p, PT &p_a, const PT &w, PT &w_a) {
  static std::stack<T> tbr;
  T a_t,a_t_a=0,a,a_a=0,b_t,b_t_a=0,b,b_a=0,aux,aux_a=0,c_t,c_t_a=0,c,c_a=0;
  if (m==mode::primal) {
    f1_t_a(m,x,x_a,x_t,x_t_a,a,a_a,a_t,a_t_a);
    f2_t_a(m,x,x_a,x_t,x_t_a,b,b_a,b_t,b_t_a);
    aux=exp(-(x-p)/w);
    c_t=-aux/w*x_t;
    c=1+aux;
    tbr.push(x_t);
    x_t=-c_t/pow(c,2);
    tbr.push(x);
    x=1/c;
    tbr.push(x_t);
    x_t=(1-x)*a_t+(b-a)*x_t+x*b_t;
    tbr.push(x);
    x=a*(1-x)+b*x;
    tbr.push(a); tbr.push(a_t);
    tbr.push(b); tbr.push(b_t);
    tbr.push(c); tbr.push(c_t);
    tbr.push(aux); 
  } else {
    aux=tbr.top(); tbr.pop();
    c_t=tbr.top(); tbr.pop();
    c=tbr.top(); tbr.pop();
    b_t=tbr.top(); tbr.pop();
    b=tbr.top(); tbr.pop();
    a_t=tbr.top(); tbr.pop();
    a=tbr.top(); tbr.pop();
    x=tbr.top(); tbr.pop();
    a_a+=(1-x)*x_a; b_a+=x*x_a; x_a=(b-a)*x_a;
    x_t=tbr.top(); tbr.pop();
    a_t_a+=(1-x)*x_t_a; b_t_a+=x*x_t_a; b_a+=x_t*x_t_a; a_a+=-x_t*x_t_a; x_a+=(b_t-a_t)*x_t_a; x_t_a=(b-a)*x_t_a;
    x=tbr.top(); tbr.pop();
    c_a+=-x_a/pow(c,2); x_a=0;
    x_t=tbr.top(); tbr.pop();
    c_t_a+=-x_t_a/pow(c,2); c_a+=2*c_t*x_t_a/pow(c,3); x_t_a=0;
    aux_a+=c_a; c_a=0;
    aux_a+=-c_t_a*x_t/w; x_t_a+=-c_t_a*aux/w; w_a+=c_t_a*aux*x_t/pow(w,2); 
    x_a+=-exp(-(x-p)/w)/w*aux_a;
    p_a+=exp(-(x-p)/w)/w*aux_a;
    w_a+=exp(-(x-p)/w)*(x-p)/pow(w,2)*aux_a; aux_a=0;
    f2_t_a(m,x,x_a,x_t,x_t_a,b,b_a,b_t,b_t_a);
    f1_t_a(m,x,x_a,x_t,x_t_a,a,a_a,a_t,a_t_a);
  }
}

template<typename T, typename PT>
void dsigmoid_dx(T &x, const PT &p, const PT &w, T &dx) {
  dx=1; sigmoid_t(x,dx,p,w);
}

template<typename T, typename PT>
void dsigmoid_dx_a(mode m, T &x, T &x_a, const PT &p, PT &p_a, const PT &w, PT &w_a, T &dx, T &dx_a) {
  static std::stack<T> tbr;
  if (m==mode::primal) {
    tbr.push(dx); 
    dx=1; sigmoid_t_a(m,x,x_a,dx,dx_a,p,p_a,w,w_a);
  } else {
    sigmoid_t_a(m,x,x_a,dx,dx_a,p,p_a,w,w_a); 
    dx=tbr.top(); tbr.pop();
    dx_a=0; 
  }
}

template<typename T, typename PT>
void newton_a(T &x, T &x_a, const T &p, T &p_a, const T &w, T &w_a, const PT &eps, const unsigned int maxit) {
  std::stack<T> tbr;
  unsigned int it=0;
  T y_a=0, dy_a=0;
  T y=x, dy;
  dsigmoid_dx_a(mode::primal,y,y_a,p,p_a,w,w_a,dy,dy_a);
  do {
    x=x-y/dy;
    tbr.push(y);
    y=x; 
    dsigmoid_dx_a(mode::primal,y,y_a,p,p_a,w,w_a,dy,dy_a);
    if (++it==maxit) break;
  } while(fabs(y)>eps);
  for (auto i=it;i>0;--i) {
    dsigmoid_dx_a(mode::adjoint,y,y_a,p,p_a,w,w_a,dy,dy_a);
    y=tbr.top(); tbr.pop(); x_a+=y_a; y_a=0;
    y_a+=-x_a/dy; dy_a+=x_a*y/pow(dy,2); 
  }
  dsigmoid_dx_a(mode::adjoint,y,y_a,p,p_a,w,w_a,dy,dy_a);
  x_a+=y_a; y_a=0;
}
