#include "Eigen/Dense"
#include <iostream>
#include <cassert>
#include <limits>
#include <cmath>

template<typename T, int N>
T f(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,N,1> &x) { 
  return pow(p.dot(x),2); 
}

template<typename T, int M, int N>
T E(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  int m=y.size();
  T o=0;
  for (auto i=0;i<m;i++) {
    Eigen::Matrix<T,N,1> xr=x.row(i);
    o+=pow(f(p,xr)-y(i),2);
  }
  return o;
}

template<typename T, int M, int N>
Eigen::Matrix<T,N,1> dEdp(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  int n=p.size();
  Eigen::Matrix<T,N,1> r(n);
  for (auto i=0;i<n;i++) {
    T dp=fabs(p(i))<1 ? sqrt(sqrt(std::numeric_limits<T>::epsilon()))
       : sqrt(sqrt(std::numeric_limits<T>::epsilon())*fabs(p(i)));
    Eigen::Matrix<T,N,1> pp=p+dp*Eigen::Matrix<T,N,1>::Unit(n,i);
    Eigen::Matrix<T,N,1> pm=p-dp*Eigen::Matrix<T,N,1>::Unit(n,i);
    r(i)=(E(pp,x,y)-E(pm,x,y))/(2*dp);
  }
  return r;
}

template<typename T, int M, int N>
Eigen::Matrix<T,N,N> ddEdpp(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  int n=p.size();
  Eigen::Matrix<T,N,N> r(n,n);
  for (auto i=0;i<n;i++) {
    T dp=fabs(p(i))<1 ? sqrt(sqrt(std::numeric_limits<T>::epsilon()))
       : sqrt(sqrt(std::numeric_limits<T>::epsilon())*fabs(p(i)));
    Eigen::Matrix<T,N,1> pp=p+dp*Eigen::Matrix<T,N,1>::Unit(n,i);
    Eigen::Matrix<T,N,1> pm=p-dp*Eigen::Matrix<T,N,1>::Unit(n,i);
    Eigen::Matrix<T,N,1> ri=(dEdp(pp,x,y)-dEdp(pm,x,y))/(2*dp);
    for (auto j=0;j<n;j++) r(j,i)=ri(j);
  }
  return r;
}

template<typename T, int M, int N>
void Newton(Eigen::Matrix<T,N,1>& p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y, const T& eps) {
  Eigen::Matrix<T,N,1> dodp=dEdp(p,x,y);
  do {
    Eigen::Matrix<T,N,N> ddodpp=ddEdpp(p,x,y);
    p+=ddodpp.llt().solve(-dodp);
    dodp=dEdp(p,x,y);
  } while (dodp.norm()>eps);
}

int main(int argc, char* argv[]) {
  assert(argc==3); int m=std::stoi(argv[1]), n=std::stoi(argv[2]);
  using T=double;
  using MT=Eigen::Matrix<T,Eigen::Dynamic,Eigen::Dynamic>;
  using VT=Eigen::Matrix<T,Eigen::Dynamic,1>;
  MT x=MT::Random(m,n); x=x.cwiseProduct(x);
  VT y=VT::Random(m), p=VT::Random(n);
  Newton(p,x,y,1e-5);
  std::cout << "p^T=" << p.transpose() << std::endl;
  std::cout << "E'^T=" << dEdp(p,x,y).transpose() << std::endl;
  std::cout << "E''=" << std::endl << ddEdpp(p,x,y) << std::endl;
  Eigen::LLT<Eigen::Matrix<T,Eigen::Dynamic,Eigen::Dynamic>> spd_test(ddEdpp(p,x,y));
  if (spd_test.info()!=Eigen::NumericalIssue)
    std::cout << "Hessian is spd." << std::endl;
  else
    std::cout << "Hessian appears to be not spd!" << std::endl;
  return 0;
}
