#include "Eigen/Dense"
#include <iostream>
#include <cassert>
#include <fstream>

template<typename T, int N>
T f(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,N,1> &x) { 
  return p.dot(x); 
}

template<typename T, int N>
Eigen::Matrix<T,N,1> dfdp(const Eigen::Matrix<T,N,1>&, const Eigen::Matrix<T,N,1> &x) { 
  return x; 
}

template<typename T, int N>
Eigen::Matrix<T,N,N> ddfdpp(const Eigen::Matrix<T,N,1>&, const Eigen::Matrix<T,N,1>& x) { 
  auto n=x.size();
  return Eigen::Matrix<T,N,N>::Zero(n,n); 
}

template<typename T, int M, int N>
T E(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  auto m=y.size();
  T o=0;
  for (auto i=0;i<m;i++) {
    Eigen::Matrix<T,N,1> xr=x.row(i);
    o+=pow(f(p,xr)-y(i),2);
  }
  return o;
}

template<typename T, int M, int N>
Eigen::Matrix<T,N,1> dEdp(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  auto m=y.size(), n=p.size();
  Eigen::Matrix<T,N,1> r=Eigen::Matrix<T,N,1>::Zero(n);
  for (auto i=0;i<m;i++) {
    Eigen::Matrix<T,N,1> xr=x.row(i);
    r+=(f(p,xr)-y(i))*dfdp(p,xr);
  }
  return 2*r;
}

template<typename T, int M, int N>
Eigen::Matrix<T,N,N> ddEdpp(const Eigen::Matrix<T,N,1> &p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y) {
  auto m=y.size(), n=p.size();
  Eigen::Matrix<T,N,N> r=Eigen::Matrix<T,N,N>::Zero(n,n);
  for (auto i=0;i<m;i++) {
    Eigen::Matrix<T,N,1> xr=x.row(i);
    r+=dfdp(p,xr)*dfdp(p,xr).transpose()+(f(p,xr)-y(i))*ddfdpp(p,xr);
  }
  return 2*r;
}

template<typename T, int M, int N>
void SteepestDescent(Eigen::Matrix<T,N,1>& p, const Eigen::Matrix<T,M,N> &x, const Eigen::Matrix<T,M,1> &y, const T& eps) {
  Eigen::Matrix<T,N,1> g=dEdp(p,x,y);
  T o=E(p,x,y),o_prev;
  while (g.norm()>eps) {
    o_prev=o;
    T alpha=2.;
    while (o_prev<=o) {
      Eigen::Matrix<T,N,1> p_trial=p;
      alpha/=2;
      p_trial-=alpha*g; o=E(p_trial,x,y);
    }
    p-=alpha*g; g=dEdp(p,x,y);
  }
}

int main(int argc, char* argv[]) {
  assert(argc==3); auto m=std::stoi(argv[1]), n=std::stoi(argv[2]);
  using T=double;
  using MT=Eigen::Matrix<T,Eigen::Dynamic,Eigen::Dynamic>;
  using VT=Eigen::Matrix<T,Eigen::Dynamic,1>;
  MT x=MT::Random(m,n); x=x.cwiseProduct(x);
  VT y=VT::Random(m);
  if (n<3) {
    std::ofstream plot_data("data");
    for (int i=0;i<m;i++) {
      for (int j=0;j<n;j++) plot_data << x(i,j) << " ";
      plot_data << y(i) << std::endl;
    }
    plot_data.close();
  }
  VT p=VT::Random(n);
  SteepestDescent(p,x,y,1e-5);
  std::cout << "p^T=" << p.transpose() << std::endl;
  std::cout << "E'^T=" << dEdp(p,x,y).transpose() << std::endl;
  std::cout << "E''=" << std::endl << ddEdpp(p,x,y) << std::endl;
  Eigen::LLT<Eigen::Matrix<T,Eigen::Dynamic,Eigen::Dynamic>> spd_test(ddEdpp(p,x,y));
  if (spd_test.info()!=Eigen::NumericalIssue)
    std::cout << "Hessian is spd." << std::endl;
  else
    std::cout << "Hessian appears to be not spd!" << std::endl;
  return 0;
}
